/*
 * Code for sending, receiving, and responding to probes
 */
#include <stdio.h>
#include <sys/time.h>
#include <netinet/in.h>

#include "libfma.h"
#include "lf_scheduler.h"
#include "lf_fms_comm.h"
#include "lf_xbar32.h"
#include "lf_myri_packet.h"
#include "lf_fma_flags.h"
#include "lf_fabric.h"

#include "libmyri.h"

#include "fma.h"
#include "fma_myri.h"
#include "fma_myri_packet.h"
#include "fma_map.h"
#include "fma_probe.h"
#include "fma_standalone.h"
#include "fma_map_fabric.h"
#include "fma_settings.h"
#include "fma_sync_timer.h"

/*
 * local prototypes
 */
static struct fma_probe_desc *fma_find_probe(unsigned int serial);
static struct fma_probe_desc *fma_get_matching_probe(int serial,
  lf_mac_addr_t *sender_mac, struct fma_nic_info *nip, int port);
static void fma_got_tagged_xbar_scout_resp(struct fma_probe_desc *pdp,
	void *pkt, int len);
static void fma_got_xbar_scout_resp(struct fma_probe_desc *pdp,
	void *pkt, int len);
static void fma_tagged_xbar_scout_failure(struct fma_probe_desc *pdp);
static void myri_xbar_scout_failure(struct fma_probe_desc *pdp);

#ifdef FM_TEST_LAG
extern unsigned char *My_lag_id;
#endif

/*
 * Handle returning probe
 */
void
fma_got_nic_scout_resp(
  struct fma_nic_info *nip,
  int port,
  struct myri_nic_scout_reply *pkt,
  int len)
{
  struct fma_nic_verify_scout_opaque_data *vsodp;
  struct fma_nic_map_scout_opaque_data *msodp;
  struct fma_nic_reply_opaque_data *rodp;
  int protocol_id;

  vsodp = (struct fma_nic_verify_scout_opaque_data *) &pkt->opaque_scout_data;
  rodp = (struct fma_nic_reply_opaque_data *) &pkt->opaque_reply_data;

  /* verify that replier is talking the right protocol */
  protocol_id = ntohs(rodp->fma_protocol_16);
  if (protocol_id != FMA_PROTOCOL_VERSION) {
    if (A.debug) {
      if (protocol_id >= 3) {
	fma_log("Dropping NIC scout reply from " LF_MAC_FORMAT
	  ", protocol ID %d != %d",
	  LF_MAC_ARGS(pkt->mac_addr), protocol_id, FMA_PROTOCOL_VERSION);
      } else {
	fma_log("Dropping NIC scout reply, protocol ID %d != %d",
	    protocol_id, FMA_PROTOCOL_VERSION);
      }
    }
    return;
  }

  /* Call the right handler based on scout type */
  if (vsodp->scout_type_8 == FMA_NST_VERIFY) {

    /* Packet has max MAC addr and nic ID - validate and if matching, convert
     * to an "origin MAC addr".  If not our packet, just drop it here.
     */
    if (LF_MAC_CMP(A.myri->my_max_mac, vsodp->origin_max_mac_addr) != 0) {
      return;
    }

    /* Our packet, convert NIC ID to MAC address and process */
    fma_got_probe_resp(nip, port, ntohl(vsodp->serial_32),
	&A.myri->nic_info[vsodp->origin_nic_8]->myri_info.mac_addr, pkt, len);

  /* it's a mapper scout - slighly different format */
  } else if (vsodp->scout_type_8 == FMA_NST_MAP
	     || vsodp->scout_type_8 == FMA_NST_REMAP) {
    msodp = (struct fma_nic_map_scout_opaque_data *) &pkt->opaque_scout_data;

    /* Packet has max MAC addr and nic ID - validate and if matching, convert
     * to an "origin MAC addr".  If not our packet, just drop it here.
     */
    if (LF_MAC_CMP(A.myri->my_max_mac, msodp->origin_max_mac_addr) != 0) {
      return;
    }

    /* pass this to the mapping code */
    fma_mf_got_nic_scout_resp(pkt, len);
  }

  /* else just drop if type bad */

}


/*
 * Handle returning xbar scout
 */
void
fma_got_xbar_scout_pkt(
  struct fma_nic_info *nip,
  int port,
  struct fma_xbar_scout_pkt *pkt,
  int len)
{
  int node_index;
  int nic_index;

  /* Packet has max MAC addr and nic ID - validate and if matching, convert
   * to an "origin MAC addr".  If not our packet, just drop it here.
   */
  if (LF_MAC_CMP(A.myri->my_max_mac, pkt->origin_max_mac_addr) != 0) {
    return;
  }

  node_index = ntohl(pkt->node_index_32);
  nic_index = LF_TOPO_NI_INDEX(node_index);

  fma_got_probe_resp(nip, port, ntohl(pkt->serial_32),
      &A.myri->nic_info[nic_index]->myri_info.mac_addr, pkt, len);
}

/*
 * Handle returning probe
 */
void
fma_got_probe_resp(
  struct fma_nic_info *nip,
  int port,
  int serial,
  lf_mac_addr_t *sender_mac,
  void *pkt,
  int len)
{
  struct fma_probe_desc *pdp;

  /* find and validate matching probe */
  pdp = fma_get_matching_probe(serial, sender_mac, nip, port);

  if (pdp == NULL) {
    return;
  }

  /* Cancel timeout timer */
  if (pdp->pd_sync_timer != NULL) {
    fma_sync_timer_cancel(pdp->pd_sync_timer);
    pdp->pd_sync_timer = NULL;
  }

  /* call completion routine */
  pdp->resp_rtn(pdp, pkt, len);
   
  /* probe is no longer active */
  fma_unlink_probe_descriptor(pdp);

  /* If not persistant, destroy the probe descriptor */
  if (!pdp->persistant) {
    LF_FREE(pdp);
  }

}

/*
 * Find a matching probe record for a packet and validate the
 * packet against it.
 */
static struct fma_probe_desc *
fma_get_matching_probe(
  int serial,
  lf_mac_addr_t *sender_mac,
  struct fma_nic_info *nip,
  int port)
{
  struct fma_probe_desc *pdp;

  /* Use the serial number to find matching probe */
  pdp = fma_find_probe(serial);
  if (pdp == NULL) {
    return NULL;
  }

  /* If it did not return the way we sent it, it doesn't count */
  if (LF_MAC_CMP(sender_mac, nip->myri_info.mac_addr) != 0
      || pdp->pd_port != port
      || pdp->origin != nip) {
    return NULL;
  }

  return pdp;
}

/*
 * Start a probe to see what's directly connected to one of our NICs.
 * We'll send a tagged xbar probe a few times, and then try an x16 probe a
 * few more times if that doesn't work.
 */
void
fma_start_direct_probe(
  struct fma_nic_info *nip,
  int port)
{
  struct fma_probe_desc *pdp;

  /* get a probe descriptor */
  pdp = fma_create_tagged_xbar_scout_probe(nip, port, FMA_PT_TAGGED_XBAR_SCOUT,
				   fma_got_tagged_xbar_scout_resp,
                                   fma_tagged_xbar_scout_failure,
				   NULL, 0);
  /* send the packet */
  fma_send_probe_packet(pdp);

  /* packet sent OK, save the probe descriptor */
  fma_link_probe_descriptor(pdp);
}

/*
 * Create a probe for scouting a tagged xbar
 */
struct fma_probe_desc *
fma_create_tagged_xbar_scout_probe(
  struct fma_nic_info *nip,
  int port,
  int type,
  fma_probe_resp_rtn_t resp_rtn,
  fma_probe_fail_rtn_t fail_rtn,
  unsigned char *route,
  int route_len)
{
  struct fma_xbar_scout_pkt *pkt;
  struct fma_probe_desc *pdp;

  /* get a probe descriptor */
  LF_CALLOC(pdp, struct fma_probe_desc, 1);
  pkt = &pdp->packet.xbar_scout_pkt;
  pdp->pkt_len = sizeof(*pkt);

  pdp->pd_serial = fma_myri_next_probe_serial();
  pdp->retries_left = FMA_XBAR_SCOUT_RETRY;
  pdp->pd_type = type;
  pdp->origin = nip;
  pdp->pd_port = port;
  pdp->pd_timeout = FMA_XBAR_SCOUT_TIMEOUT;
  pdp->resp_rtn = resp_rtn;
  pdp->fail_rtn = fail_rtn;

  /* create loop route */
  pdp->pd_route_len = lf_loop_route(pdp->pd_route, route, route_len,
      LF_TAGGED_XBAR_QUERY_ROUTE);

  /* Fill in a tagged xbar scout packet */
  fma_fill_tagged_xbar_scout(pkt, nip, port, pdp->pd_serial);

  return pdp;

 except:
  fma_perror_exit(1);
  return NULL;
}

/*
 * Create a probe for scouting a generic xbar
 */
struct fma_probe_desc *
fma_create_xbar_scout_probe(
  struct fma_nic_info *nip,
  int port,
  int type,
  fma_probe_resp_rtn_t resp_rtn,
  fma_probe_fail_rtn_t fail_rtn,
  unsigned char *route,
  int route_len)
{
  struct fma_xbar_scout_pkt *pkt;
  struct fma_probe_desc *pdp;

  /* get a probe descriptor */
  LF_CALLOC(pdp, struct fma_probe_desc, 1);
  pkt = &pdp->packet.xbar_scout_pkt;
  pdp->pkt_len = sizeof(*pkt);

  pdp->pd_serial = fma_myri_next_probe_serial();
  pdp->retries_left = FMA_XBAR_SCOUT_RETRY;
  pdp->pd_type = type;
  pdp->origin = nip;
  pdp->pd_port = port;
  pdp->pd_timeout = FMA_XBAR_SCOUT_TIMEOUT;
  pdp->resp_rtn = resp_rtn;
  pdp->fail_rtn = fail_rtn;

  /* create loop route */
  pdp->pd_route_len = lf_loop_route(pdp->pd_route, route, route_len,
      LF_DELTA_TO_ROUTE(0));

  /* Fill in an xbar scout packet */
  fma_fill_xbar_scout(pkt, nip, port, pdp->pd_serial);

  return pdp;

 except:
  fma_perror_exit(1);
  return NULL;
}

/*
 * Create a probe for scouting for a NIC
 */
struct fma_probe_desc *
fma_create_nic_scout_probe(
  struct fma_nic_info *nip,
  int port,
  int type,
  fma_probe_resp_rtn_t resp_rtn,
  fma_probe_fail_rtn_t fail_rtn,
  unsigned char *route,
  int route_len)
{
  struct myri_nic_scout *pkt;
  struct fma_probe_desc *pdp;
  struct fma_settings *asp;

  asp = A.settings;

  /* use the route we were passed */
  if (route_len > MYRI_MAX_ROUTE_LEN) {
    LF_ERROR(("Route too long in fma_probe"));
  }

  /* get a probe descriptor */
  LF_CALLOC(pdp, struct fma_probe_desc, 1);

  pdp->pd_serial = fma_myri_next_probe_serial();
  pdp->retries_left = asp->nic_scout_retries;
  pdp->pd_type = type;
  pdp->origin = nip;
  pdp->pd_port = port;
  pdp->pd_timeout = asp->nic_scout_timeout;
  pdp->resp_rtn = resp_rtn;
  pdp->fail_rtn = fail_rtn;

  /* just copy the route */
  memcpy(pdp->pd_route, route, route_len);
  pdp->pd_route_len = route_len;

  /* Fill in a NIC scout packet */
  pkt = &pdp->packet.nic_scout;
  fma_fill_nic_verify_scout(pkt, nip, port, pdp->pd_serial, route, route_len);
  pdp->pkt_len = sizeof(*pkt);

  return pdp;

 except:
  fma_perror_exit(1);
  return NULL;
}



/*
 * Fill in Myrinet packet for a tagged xbar probe
 */
void
fma_fill_tagged_xbar_scout(
  struct fma_xbar_scout_pkt *pkt,
  struct fma_nic_info *nip,
  int port,
  int serial)
{
  /* mostly the same as filling a regular xbar scout */
  fma_fill_xbar_scout(pkt, nip, port, serial);

  /* Change the type and zero-out the tagged xbar portion */
  pkt->hdr.subtype_16 = htons(FMA_SUBTYPE_TAGGED_XBAR_SCOUT);
  memset(&pkt->tagged_xbar_insert, 0, sizeof(struct lf_tagged_xbar_insert));
}

/*
 * Fill in Myrinet packet for an xbar probe
 */
void
fma_fill_xbar_scout(
  struct fma_xbar_scout_pkt *pkt,
  struct fma_nic_info *nip,
  int port,
  int serial)
{
  int node_index;

  pkt->hdr.type_16 = htons(FMA_PACKET_TYPE);
  pkt->hdr.subtype_16 = htons(FMA_SUBTYPE_XBAR_SCOUT);
  pkt->serial_32 = htonl(serial);

  node_index = LF_TOPO_NIC_INDEX(nip->nic_index, port);
  pkt->node_index_32 = htonl(node_index);
  LF_MAC_COPY(pkt->origin_max_mac_addr, A.myri->my_max_mac);
}

/*
 * Fill in Myrinet packet for a NIC verify scout
 */
void
fma_fill_nic_verify_scout(
  struct myri_nic_scout *pkt,
  struct fma_nic_info *nip,
  int port,
  int serial,
  unsigned char *route,
  int route_len)
{
  struct fma_nic_verify_scout_opaque_data *sodp;

  sodp = (struct fma_nic_verify_scout_opaque_data *) &pkt->opaque_scout_data;

  pkt->hdr.type_16 = htons(MYRI_PACKET_TYPE);
  pkt->hdr.subtype_16 = htons(MYRI_SUBTYPE_NIC_SCOUT);

  /* fill in opaque fields */
  sodp->scout_type_8 = FMA_NST_VERIFY;
  LF_MAC_COPY(sodp->origin_max_mac_addr, A.myri->my_max_mac);
  sodp->origin_nic_8 = nip->nic_index;
  sodp->origin_port_8 = port;
  sodp->serial_32 = htonl(serial);
  sodp->fma_protocol_16 = htons(FMA_PROTOCOL_VERSION);

  /* Fill in our mapping info */
  sodp->level_8 = A.stand->my_level;

  /* Fill in the generic parts */
  fma_fill_nic_generic_scout(pkt, route, route_len);
}

/*
 * Fill in the generic part of a Myrinet packet for a NIC scout
 */
void
fma_fill_nic_generic_scout(
  struct myri_nic_scout *pkt,
  unsigned char *route,
  int route_len)
{
  int i;

  pkt->hdr.type_16 = htons(MYRI_PACKET_TYPE);
  pkt->hdr.subtype_16 = htons(MYRI_SUBTYPE_NIC_SCOUT);

  /* copy over the reverse route */
  for (i=0; i<route_len; ++i) {
    pkt->reply_route[route_len-i-1] =
	LF_DELTA_TO_ROUTE(-LF_ROUTE_TO_DELTA(route[i]));
  }
  pkt->reply_route_len_8 = route_len;
  pkt->nic_replied_8 = 0;		/* make sure */
}

/*
 * Send the packet for a probe
 */
void
fma_send_probe_packet(
  struct fma_probe_desc *pdp)
{
  /* safety check */
  if (pdp->pd_sync_timer != NULL) {
    LF_ERROR(("pd_sync_timer should not be NULL here"));
  }

  /* send the packet using info from probe descriptor */
  pdp->pd_sync_timer = fma_myri_raw_send_with_timer(pdp->origin->nic_handle,
	  pdp->pd_port, pdp->pd_route, pdp->pd_route_len, &pdp->packet,
	  pdp->pkt_len, fma_probe_timeout, pdp, pdp->pd_timeout);
  if (pdp->pd_sync_timer == NULL) LF_ERROR(("Error sending packet"));

  return;
 except:
  fma_perror_exit(1);
}

/*
 * a tagged xbar scout sequence failed - switch to scouting for xbar 16s.
 */
static void
fma_tagged_xbar_scout_failure(
  struct fma_probe_desc *pdp)
{
  struct fma_probe_desc *npdp;

  /* switch to looking for xbar with no ID */
  npdp = fma_create_xbar_scout_probe(pdp->origin, pdp->pd_port,
      				   FMA_PT_XBAR_SCOUT,
				   fma_got_xbar_scout_resp,
                                   myri_xbar_scout_failure,
				   NULL, 0);

  fma_send_probe_packet(npdp);	/* send the packet */

  /* packet sent OK, save the probe descriptor */
  fma_link_probe_descriptor(npdp);
}

/*
 * an xbar scout timed out hard
 */
static void
myri_xbar_scout_failure(
  struct fma_probe_desc *pdp)
{
  /* XXX - need more variety here */

  /* mark connection "unknown" and give up */
  pdp->origin->ni_xbar_id[pdp->pd_port] = -1;
  fma_note_query_completion();
}

/*
 * Got tagged xbar scout response
 */
static void
fma_got_tagged_xbar_scout_resp(
  struct fma_probe_desc *pdp,
  void *vpkt,
  int len)
{
  struct fma_xbar_scout_pkt *pkt;
  int xbar_id;
  int xbar_port;

  pkt = vpkt;

  /* get xbar ID and port from packet */
  xbar_id = lf_swap_l(ntohl(pkt->tagged_xbar_insert.id));
  xbar_port = pkt->tagged_xbar_insert.absolute_port;

  /* Fill in the xbar info for this NIC */
  pdp->origin->ni_xbar_id[pdp->pd_port] = xbar_id;
  pdp->origin->ni_xbar_port[pdp->pd_port] = xbar_port;

  /* Notify of query completion */
  fma_note_query_completion();

  return;
}

/*
 * Got an xbar scout response
 */
static void
fma_got_xbar_scout_resp(
  struct fma_probe_desc *pdp,
  void *vpkt,
  int len)
{
  struct fma_xbar_scout_pkt *pkt;

  pkt = vpkt;

  /* Fill in the xbar info for this NIC */
  pdp->origin->ni_xbar_id[pdp->pd_port] = 0;

  /* Notify of query completion */
  fma_note_query_completion();

  return;
}

/*
 * Find a probe by serial number
 */
static struct fma_probe_desc *
fma_find_probe(
  unsigned int serial)
{
  struct fma_probe_desc *pdp;
  struct fma_probe_desc *anchor;
  int index;

  index = FMA_PROBE_HASH(serial);

  anchor = &A.myri->probe_anchors[index];
  for (pdp=anchor->pd_next; pdp!=anchor; pdp=pdp->pd_next) {
    if (pdp->pd_serial == serial) {
      return pdp;
    }
  }
  return NULL;
}

/*
 * Got an low-level NIC scout
 * We need to do different things with this based on what the reason is for 
 * this scout.
 * FMA_NST_MAP - scouting
 * FMA_NST_VERIFY - verifying the fabric, just reply.
 */
void
fma_got_nic_scout(
  struct fma_nic_info *nip,
  int port,
  struct myri_nic_scout *pkt,
  int len)
{
  struct myri_nic_scout_reply resp;
  struct myri_nic_scout_reply *rp;
  struct fma_nic_verify_scout_opaque_data *vsodp;
  struct fma_nic_map_scout_opaque_data *msodp;
  struct fma_nic_reply_opaque_data *rodp;
  lf_mac_addr_t *max_mac_addr;
  int protocol_id;
  enum fma_nic_scout_type scout_type;
  int level;
  int rc;

  vsodp = (struct fma_nic_verify_scout_opaque_data *) &pkt->opaque_scout_data;
  msodp = (struct fma_nic_map_scout_opaque_data *) &pkt->opaque_scout_data;
  scout_type = vsodp->scout_type_8;

  /* Drop if not matching protocol */
  protocol_id = ntohs(vsodp->fma_protocol_16);
  if (protocol_id != FMA_PROTOCOL_VERSION) {
    if (A.debug) {
      if (protocol_id >= 3) {
	unsigned char *mac_addr;

	if (scout_type == FMA_NST_VERIFY) {
	  mac_addr = vsodp->origin_max_mac_addr;
	} else {
	  mac_addr = msodp->origin_max_mac_addr;
	}

	fma_log("Dropping NIC scout from " LF_MAC_FORMAT
	  ", protocol ID %d != %d",
	  LF_MAC_ARGS(mac_addr), protocol_id, FMA_PROTOCOL_VERSION);
      } else {
	fma_log("Dropping NIC scout, protocol ID %d != %d",
	    protocol_id, FMA_PROTOCOL_VERSION);
      }
    }
    return;
  }

  if (scout_type == FMA_NST_VERIFY) {
    level = vsodp->level_8;
    max_mac_addr = &vsodp->origin_max_mac_addr;

    if (A.debug > 2) {
      fma_log("got verify scout from %s, NIC %d, p%d, reason=%d,"
	      "lev=%d, serial=%d",
	      fma_mac_to_hostname(vsodp->origin_max_mac_addr),
	      vsodp->origin_nic_8,
	      vsodp->origin_port_8, vsodp->scout_type_8, vsodp->level_8,
	      ntohl(vsodp->serial_32));
    }

  } else {
    level = msodp->level_8;
    max_mac_addr = &msodp->origin_max_mac_addr;
  }

#if 0
/* XXX - if we are permanent standalone, ignore this  packet */
 if (num_servers <= 0 && level >= FMA_MAX_MAPPER_LEVEL)
  return;
#endif

  /* tell standalone that we got a scout */
  if (A.run_state == FMA_RUN_STANDALONE) {

    fma_standalone_got_contact(level, *max_mac_addr,
			       nip, port,
			       pkt->reply_route, pkt->reply_route_len_8,
			       scout_type);

  /* If not in standalone mode and we get a non-FMS level mapping packet,
   * then mapping needs to restart.
   */
  } else if (A.run_state != FMA_RUN_BOOTING
             && scout_type != FMA_NST_VERIFY
             && level < FMA_FMS_MAPPER_LEVEL) { 
    lf_string_t why;

    /* XXX - what do we do here if in PENDING state?? */

    sprintf(why, "Got non-FMS mapping packet from %s, NIC %d, p%d, level=%d",
	    fma_mac_to_hostname(vsodp->origin_max_mac_addr),
	    vsodp->origin_nic_8, vsodp->origin_port_8, level);
    fma_map_is_invalid(TRUE, why);
  }

  /* At this point, if NIC already replied, we are done */
  if (pkt->nic_replied_8) {
    if (A.debug > 1) fma_log("NIC replied for us");
    return;
  }

  /* fill in type and subtype */
  rp = &resp;
  rp->hdr.type_16 = htons(MYRI_PACKET_TYPE);
  rp->hdr.subtype_16 = htons(MYRI_SUBTYPE_NIC_SCOUT_REPLY);

  /* copy back the querier's opaque data */
  memcpy(&rp->opaque_scout_data, &pkt->opaque_scout_data,
      MYRI_NIC_SCOUT_OPAQUE_SIZE);

  /* Fill in the myri-generic reply data */
  LF_MAC_COPY(rp->mac_addr, nip->myri_info.mac_addr);
  rp->port_8 = port;
  rp->num_ports_8 = nip->myri_info.num_ports;
  rp->firmware_type_8 = myri_firmware_type();

  /* Fill in FMA-specific stuff */
  rodp = (struct fma_nic_reply_opaque_data *) &resp.opaque_reply_data;
  memcpy(rodp, nip->nic_reply_data, sizeof(*rodp));

#ifdef FM_TEST_LAG
  if (My_lag_id != NULL) {
    strcpy(rodp->lag_id, My_lag_id);
  }
#endif

  rc = fma_myri_raw_send(nip->nic_handle, port,
                     pkt->reply_route, pkt->reply_route_len_8,
                     &resp, sizeof(resp), NULL, NULL);
  if (rc != 0) LF_ERROR(("Error sending scout reply"));

  return;

 except:
  fma_perror_exit(1);
}

/*
 * Generic timeout handler for probes - if retry count not exhausted, re-send
 * the packet, else call the failure routine for the probe
 */
void
fma_probe_timeout(
  void *vpdp)
{
  struct fma_probe_desc *pdp;

  pdp = vpdp;
  pdp->pd_sync_timer = NULL;		/* expired */

  /* If any retries left, re-send packet with the same serial number */
  if (pdp->retries_left-- > 0) {

    fma_send_probe_packet(pdp);	/* send the packet */

  /* No retries remaining, call failure routine */
  } else {

    pdp->fail_rtn(pdp);

    fma_unlink_probe_descriptor(pdp);
    if (!pdp->persistant) {
      LF_FREE(pdp);
    }
  }
}

/*
 * set the current level for all probes
 */
void
fma_set_probe_levels(
  int level)
{
  struct fma_probe_desc *pdp;
  struct fma_probe_desc *anchor;
  struct fma_nic_verify_scout_opaque_data *sodp;

  anchor = A.myri->verify_anchor;
  for (pdp=anchor->user_next; pdp!=anchor; pdp=pdp->user_next) {
    if (pdp->packet.nic_scout.hdr.subtype_16 == htons(MYRI_SUBTYPE_NIC_SCOUT)) {
      sodp = (struct fma_nic_verify_scout_opaque_data *)
	      &pdp->packet.nic_scout.opaque_scout_data;
      sodp->level_8 = level;
    }
  }
}
